// ============================================================================
// ============================================================================
// ============================================================================
// ==                                                                        ==
// == Name    : TheEmuLib.Emu_Tenter_XY.04.fsh                               ==
// == Type    : Fragment shader                                              ==
// == Version : 1.0.5 (2017/01/20)                                           ==
// == Creator : TheEmu © TheEmu 2017, Some Rights Reserved                   ==
// == Licence : Creative Commons Attribution-ShareAlike 4.0                  ==
// ==           http://creativecommons.org/licences/by-sa/4.0                ==
// ==                                                                        ==
// == Purpose: To apply a 2D "tentering" transform to an image.              ==
// ==                                                                        ==
// == Description: The source image is distorted using the tenter transform. ==
// == Tentering is the process of stretching fabric by pulling it at it at a ==
// == set of points at its edges often using a set of tenter hooks to attach ==
// == it to the cords that do the pulling.  This is the origin of the phrase ==
// == "to be on tenterhooks". The transform used here applies a similar kind ==
// == of stretching and I have named it accordingly. TheEmu.                 ==
// ==                                                                        ==
// == A number of versions of the tentering shader are provided which differ ==
// == only in the maximum number of stretches that they support. Although it ==
// == is always possible to use the version supporting the highest number of ==
// == stretches it is preferable to use the smallest one that  supports  the ==
// == the required number as this will reduce the GPU loading.               ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == This file is a member of The Emu's shader library.                     ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == Update history:                                                        ==
// ==                                                                        ==
// ==   2017/01/12 - v1.0.0 - Initial version.                               ==
// ==   2017/01/14 - v1.0.1 - Improved efficiency of Stretch_1D.             ==
// ==   2017/01/17 - v1.0.2 - Added edge mode 2 (curved and pointy)          ==
// ==   2017/01/19 - v1.0.3 - Removed unused argument from Stretch_1D        ==
// ==   2017/01/20 - v1.0.4 - Added edge modes 3 and 4 (quadratic)           ==
// ==   2017/01/20 - v1.0.5 - Changed stretch parameters to be vec4          ==
// ==                                                                        ==
// ============================================================================
// ============================================================================
// ============================================================================

// ============================================================================
// == Standard shader inputs ==================================================
// ============================================================================

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// The image that is to be manipulated.

uniform sampler2D iChannel0;

// ============================================================================
// == Imports from TheEmuLib ==================================================
// ============================================================================
//
// The GLSL shader language currently provides no mechanism for importing  any
// elements that are defined in other modules, not even C's crude source level
// #include mechanism. In the absence of anything better TheEmuLib handles any
// imports by manualy copying relevent utility code snippets from the  sources
// in the Shader Lib.Inc directory. This is very crude but I have attempted to
// be systematic in the way in which this is presented in the library sources.
//
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Basic_Constants.lib.src

#define PI  3.1415926535897932384626433832795
#define TAU 6.2831853071795864769252867665590

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Common_Utilities.lib.src

#define EMU_DEFAULT(type,x,default_value) ( (x==type(0.0)) ? (default_value) : (x) )

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Functions from TheEmuLib.Emu_Coordinate_Normalisation.lib.src

#define EMU_NORMALISE_TO_WINDOW_1(xy,wsize) ( (xy)/(wsize) )

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

vec2 Emu_Normalise_to_Window ( vec2 xy )
 { return EMU_NORMALISE_TO_WINDOW_1 ( xy, u_WindowSize.xy );
 }

// ============================================================================
// == Shader specific inputs ==================================================
// ============================================================================

// The transformed image can be optionaly scaled and its "hotspot" moved.  The
// scale factor affects the image size with scale factors greater than  having
// the effect of enlarging the image while factors between 0.0 and 1.0  reduce
// it. Negative scale factors are allowed.
//
// The hotspot species the point in the image that acts as the origin for  its
// local coordinate system.  Values  for  each component range from 0.0 to 1.0
// if the hotspot lies with the image,  but  values  outside of this range may
// be used. For the tentering shaders the main effect is to relocate the image
// with a hotspot 0.5, 0.5 causing it to be centered in the graphics window.

uniform vec2 Emu_Tenter_XY_scale;
uniform vec2 Emu_Tenter_XY_hotspot;

vec2 scale   = EMU_DEFAULT ( vec2, Emu_Tenter_XY_scale,   vec2(1.0) );
vec2 hotspot = EMU_DEFAULT ( vec2, Emu_Tenter_XY_hotspot, vec2(0.0) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default the angles that are induced into the edges where they are pulled
// By default the resulting shape will be polygonal having straight sides and
// angular corners.  This can be modified using the following parameter which
// may have the following values
// 
//         0 - Straight sides, angular, default
//         1 - Curved sides, smooth corners
//         2 - Curved sides, pointy corners

uniform int Emu_Tenter_XY_edge_mode;

#define edge_mode Emu_Tenter_XY_edge_mode

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The tentering operation is described by a series of X and Y stretches which
// are defined by the stretch_Xnn and stretch_Ynn parameters.  Each  of  these
// define a pair of points on opposite edges of the source image and how  much
// each is to be moved by the stretching.  The stretches are parallel to the X
// and Y directions with the stretch_Xnn parameters defining those parallel to
// the X axis and stretch_Ynn those parallel to the Y axis.
//
// The components of each of the stretch_Xnn parameters are
//
//    The position of the stretch in the Y direction
//    Displacement in the X direction at the Y=0 edge
//    Displacement in the X direction at the Y=1 edge
//    A spare parameter reserved for future use
//
// and those for the stretch_Ynn parameters are
//
//    The position of the stretch in the X direction
//    Displacement in the Y direction at the X=0 edge
//    Displacement in the Y direction at the X=1 edge
//    A spare parameter reserved for future use
//
// A positive value indicate that a displacement is directed outward, negative
// values may be used to acheive compression rather than stretching. Positions
// may range from 0.0 to 1.0 corresponding to opposite edges of the image that
// is being manipulated.
//
// It is not necessary to supply the full set of stretch parameters, but those
// that are defined should be such that if N stretches are defined for X or Y
//
//      Stretches 1 to N are defined
//      Stretches N+1 onwards are not defined
//      The stretch positions increase with nn
//
// If these restrictions are not obeyed the effect of the shader is undefined.

uniform vec4 Emu_Tenter_XY_stretch_X01;
uniform vec4 Emu_Tenter_XY_stretch_X02;
uniform vec4 Emu_Tenter_XY_stretch_X03;
uniform vec4 Emu_Tenter_XY_stretch_X04;

#define  N_stretch_X 4 // Number of stretch_X parameters

uniform vec4 Emu_Tenter_XY_stretch_Y01;
uniform vec4 Emu_Tenter_XY_stretch_Y02;
uniform vec4 Emu_Tenter_XY_stretch_Y03;
uniform vec4 Emu_Tenter_XY_stretch_Y04;

#define  N_stretch_Y 4 // Number of stretch_Y parameters

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The stretch parameters are collected into the following arrays. The extra
// elements define the corners as not being moved by the transform and their
// presence in the arrays simplifies the logic in the body of the shader.

vec4 stretch_X[] = vec4[N_stretch_X+2]
 ( vec4 ( 0.0, 0.0, 0.0, 0.0 ),
   EMU_DEFAULT ( vec4, Emu_Tenter_XY_stretch_X01, vec4(1.0,0.0,0.0,0.0) ),
   EMU_DEFAULT ( vec4, Emu_Tenter_XY_stretch_X02, vec4(1.0,0.0,0.0,0.0) ),
   EMU_DEFAULT ( vec4, Emu_Tenter_XY_stretch_X03, vec4(1.0,0.0,0.0,0.0) ),
   EMU_DEFAULT ( vec4, Emu_Tenter_XY_stretch_X04, vec4(1.0,0.0,0.0,0.0) ),
   vec4 ( 1.0, 0.0, 0.0, 0.0 )
 );

vec4 stretch_Y[] = vec4[N_stretch_Y+2]
 ( vec4 ( 0.0, 0.0, 0.0, 0.0 ),
   EMU_DEFAULT ( vec4, Emu_Tenter_XY_stretch_Y01, vec4(1.0,0.0,0.0,0.0) ),
   EMU_DEFAULT ( vec4, Emu_Tenter_XY_stretch_Y02, vec4(1.0,0.0,0.0,0.0) ),
   EMU_DEFAULT ( vec4, Emu_Tenter_XY_stretch_Y03, vec4(1.0,0.0,0.0,0.0) ),
   EMU_DEFAULT ( vec4, Emu_Tenter_XY_stretch_Y04, vec4(1.0,0.0,0.0,0.0) ),
   vec4 ( 1.0, 0.0, 0.0, 0.0 )
 );

// ============================================================================
// == Internal support functions ==============================================
// ============================================================================

vec2 quad_stretch ( vec2 xy, float f1, vec4 s0, vec4 s1, float qq )
 {
   vec2 f2 = vec2(f1);

   if ( abs(s1.y) > abs(s0.y) ) f2.x = 1.0 - f2.x;
   if ( abs(s1.z) > abs(s0.z) ) f2.y = 1.0 - f2.y;

   f2 = f2 - qq; // Optionaly invert the form of the stretch.

   f2 = f2 * f2;

   if ( abs(s1.y) > abs(s0.y) ) f2.x = 1.0 - f2.x;
   if ( abs(s1.z) > abs(s0.z) ) f2.y = 1.0 - f2.y;

   return f2;
 }

// ============================================================================
// == The shader's major functions ============================================
// ============================================================================

// Stretch_1D - Performs a one dimensional stretch at point xy. The stretch is
// defined by the two stretch descriptors, s0 and s1, where s1 is that for the
// the current stretch and s0 for the previous stretch.

float Stretch_1D ( vec2 xy, vec4 s0, vec4 s1 )
 {
   // Determine how far, as a fraction, we are into the s1 region. N.B.
   // the "x" components of the stretch parameters are its position in
   // the direction perpendicular to the stretch direction - hence the
   // somewhat odd looking mixture of xy.y and s0.x used here.

   float f1 = ( xy.y - s0.x ) / ( s1.x - s0.x );
   if ( f1 != fract(f1) ) return 0.0;

   // Optionaly cause the edges of the stretched image to be curved.

   vec2 f2;

   switch ( edge_mode )
    { 
      case 0:  f2 = vec2 ( f1 );                              break;
      case 1:  f2 = vec2 ( smoothstep(0.0,1.0,f1) );          break;
      case 2:  f2 = vec2 ( f1*2.0 - smoothstep(0.0,1.0,f1) ); break;
      case 3:  f2 = quad_stretch ( xy, f1, s0, s1, 0.0 );     break;
      case 4:  f2 = quad_stretch ( xy, f1, s0, s1, 1.0 );     break;
      default: f2 = vec2(0.0);
    }

   // Interpolate to get the two edge displacements that are applicable
   // to points that lying on the line in the stretch direction passing
   // through the current point. The edge displacement ranges from that
   // for s0 at one end of the range covered by s1 upto the maximum for
   // s1 at the other end.
   
   vec2 dd = mix ( s0.yz, s1.yz, f2 );

   // Interpolate the stretch along the line in the stretch  direction,
   // but only apply this stretch to the appropriate part of the image.

   return mix ( dd.x, dd.y, (xy.x-dd.x)/(1.0-dd.x+dd.y) );

 }

// ============================================================================

vec4 Emu_Tenter_XY ( vec2 uv )
 {
   // Apply the transform with its fixed point at the hotspot.  Note, the code
   // here is the inverse of the required transform because we are determining
   // what point in the source image will be transformed to the current pixel.

   vec2 st = ( uv - hotspot ) / scale + hotspot;
   vec2 dd = vec2(0.0);

   // We could use a loop here, which would be neater and would avoid having
   // to change this bit of code if the number of stretch parameters is ever
   // changed. However, when I used a loop it ran slowly, presumably because
   // the individual stretches can run in parallel here but not when using a
   // loop.  The  slowdown due to using a loop may be compiler dependant but
   // the easiest thing is to unroll the loop and use the following code.

   dd.x = Stretch_1D ( st.xy, stretch_X[ 0], stretch_X[ 1] )
        + Stretch_1D ( st.xy, stretch_X[ 1], stretch_X[ 2] )
        + Stretch_1D ( st.xy, stretch_X[ 2], stretch_X[ 3] )
        + Stretch_1D ( st.xy, stretch_X[ 3], stretch_X[ 4] )
        + Stretch_1D ( st.xy, stretch_X[ 4], stretch_X[ 5] );

   dd.y = Stretch_1D ( st.yx, stretch_Y[ 0], stretch_Y[ 1] )
        + Stretch_1D ( st.yx, stretch_Y[ 1], stretch_Y[ 2] )
        + Stretch_1D ( st.yx, stretch_Y[ 2], stretch_Y[ 3] )
        + Stretch_1D ( st.yx, stretch_Y[ 3], stretch_Y[ 4] )
        + Stretch_1D ( st.yx, stretch_Y[ 4], stretch_Y[ 5] );

   st = st - dd;

   // Determine the colour using transparent black if the point
   // lies outside of the area of the transformed source image.

   return ( st == fract(abs(st)) ) ? texture2D(iChannel0,st) : vec4(0.0);

 }

// ============================================================================
// == The shader's main routine ===============================================
// ============================================================================

void main ( void )
 {
   // Get the normalised coordinates of the current point.

   vec2 uv = Emu_Normalise_to_Window ( gl_FragCoord.xy );

   // Apply the transform and update the shader's outputs.

   gl_FragColor = Emu_Tenter_XY(uv) * gl_Color;

 }

// ============================================================================
